﻿/*
 i-net software provides programming examples for illustration only, without warranty
 either expressed or implied, including, but not limited to, the implied warranties
 of merchantability and/or fitness for a particular purpose. This programming example
 assumes that you are familiar with the programming language being demonstrated and
 the tools used to create and debug procedures. i-net software support professionals
 can help explain the functionality of a particular procedure, but they will not modify
 these examples to provide added functionality or construct procedures to meet your
 specific needs.
  
 © i-net software 1998-2013

*/
using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Drawing;
using System.Data;

using System.Text;
using System.Windows.Forms;
using System.IO;
using System.Collections;
using System.Threading;
using System.Globalization;
using System.Web;
using System.Drawing.Drawing2D;
using System.Diagnostics;
using System.Drawing.Printing;
using System.Net;
using Inet.Viewer.Data;
using Inet.Viewer.WinForms.Prompt;
using Inet.Viewer.WinForms.Export;


namespace Inet.Viewer.WinForms
{
    /// <summary>
    /// The ReportViewer class is a Control to show the .Net Version of the viewer for i-net Clear Reports server. The Report Viewer is derived from UserControl and can be used in Visual 
    /// Studio via the Toolbox. 
    /// </summary>
    /// <remarks>
    /// <b>How to use the ReportViewer:</b>
    /// Add an instance of the ReportViewer to a Form or a Panel. To show one report you need to add a  <see cref="Inet.Viewer.IReportView"/>
    /// to the ReportViewer. For the ReportView you have to add an implemenation of the Interface  <see cref="Inet.Viewer.IRenderData"/>.
    /// Currently there are two implementations of the interface.  implementation for access to a Clear Reports Server.
    ///       
    ///      <code>
    ///      // Example for the DataServer
    ///      string url = "http://localhost:9000/?report=file:C:/Reports/Testreport.rpt";
    ///      // generate the ReportData
    ///      IReportData data = new URLRenderData(url);            
    ///      
    ///      // create the view
    ///      ReportView view = new ReportView(data);            
    ///
    ///      // add view to the ReportViewer
    ///      this.reportViewer1.AddReportView(view);           
    ///      </code>
    /// </remarks>
    /// <seealso cref="Inet.Viewer.IRenderData"/>  
    [ToolboxItem(true)]
    [ToolboxBitmap(typeof(ReportViewer), "Viewer")]
    //[ProvideToolboxControl("i-net software", false)]
    public partial class ReportViewer : UserControl, IReportViewer
    {

        private const int TabCloseSelectedTop = 4;
        private const int TabCloseSelectedRight = 20;
        private const int TabCloseImageSize = 16;
        internal const string ReportViewPanelName = "ReportViewPanel";

        private Dictionary<IReportView, TabPage> viewToPageCollection = new Dictionary<IReportView, TabPage>();
        private Dictionary<TabPage, IReportView> pageToViewCollection = new Dictionary<TabPage, IReportView>();
        private Image closeButton;
        private Image closeButtonDisabled;

        /// <summary>
        /// Used for the Property CurrentView
        /// </summary>
        private IReportView currentView;

        /// <summary>
        /// For the Property ReportViews
        /// </summary>
        private List<IReportView> reportViews;

        private System.Drawing.Printing.PrinterSettings printerSettings = new PrinterSettings();
        private PromptPanelBuilder promptPanelBuilder;
        private ExportDialog exportDialog = new ExportDialog();

        /// <summary>
        /// Initializes the ReportViewer, adds all the controls and sets the label according to the current CultureInfo
        /// </summary>
        public ReportViewer()
        {
            closeButton = Images.closeButtonImage;
            closeButtonDisabled = Images.closeButtonDisabledImage;
            promptPanelBuilder = new PromptPanelBuilder();
            InitializeComponent();
            ReportViews = new List<IReportView>();
            ShowToolbar = true;
            searchPanel.ReportViewer = this;
        }

        /// <summary>
        /// This property shows or hides the Toolbar
        /// </summary>
        public bool ShowToolbar
        {
            get { return ToolBar.Visible; }
            set { ToolBar.Visible = value; }
        }

        /// <summary>
        /// This List holds the added ReportViews
        /// </summary>
        private List<IReportView> ReportViews
        {
            get
            {
                return reportViews;
            }
            set
            {
                reportViews = value;
            }
        }


        /// <summary>
        /// Set the View that is currently shown.
        /// Only a view that was added before will be shown! A view that is was not added before will be ignored        
        /// <inheritdoc/>        
        /// </summary>
        public IReportView CurrentReportView
        {
            get
            {
                return currentView;
            }
            set
            {
                IReportView oldCurrentView = currentView;
                if (value != null && ReportViews.Contains(value))
                {
                    currentView = value;
                    // Get TabPage for ReportView                    
                    TabPage page = this.viewToPageCollection[currentView];
                    this.tabControl1.SelectedTab = page;

                }
                else
                {
                    // clear the old tab     
                    currentView = null;

                }

                // Only if changed
                if (oldCurrentView != currentView)
                {
                    this.ToolBar.ReportView = currentView;
                    OnViewChanged();
                }
            }
        }

        /// <summary>
        /// Called when the view was changes. Fires the ViewChanged event.
        /// </summary>
        public void OnViewChanged()
        {
            if (ViewChanged != null)
            {
                ViewChanged(this, new EventArgs());
            }
        }


        /// <summary>
        /// Add report and set it as CurrentView.
        /// If the view is null it will be ignored.        
        /// <exception cref="ArgumentException"> Argument exception if a ReportView instacne is added a second time</exception>
        /// </summary>
        /// <param name="view">The <see cref="Inet.Viewer.IReportView"/> that shall be displayed in the ReportViewer</param>
        public void AddReportView(IReportView view)
        {
            ReportView rptView = view as ReportView;
            if (rptView != null)
            {
                TabPage page = null;
                try
                {
                    // Create a Tab for this view     
                    rptView.TabPage = page = new TabPage();
                    rptView.ReportViewer = this;
                    page.Text = " ... ";
                    page.ImageIndex = 0;// this image needs to be set so the width of the tabs is correct               
                    this.tabControl1.TabPages.Add(page);

                    page.Controls.Add(rptView);


                    // Add view to the List
                    ReportViews.Add(rptView);

                    // Add the relationships
                    viewToPageCollection.Add(rptView, page);
                    pageToViewCollection.Add(page, rptView);

                    // set the tab and the view
                    CurrentReportView = rptView;
                }

                catch (Exception e)
                {
                    // if an exception occurs here, remove the TabPage and view, if it was added
                    if (page != null)
                    {
                        this.tabControl1.TabPages.Remove(page);
                        pageToViewCollection.Remove(page);
                    }

                    ReportViews.Remove(view);
                    viewToPageCollection.Remove(view);

                    if (e is ArgumentException)
                    {
                        throw e;
                    }
                    else
                    {
                        throw new ViewerException("Something went wrong adding the ReportView to the ReportViewer", e);
                    }
                }
            }
        }

        /// <summary>
        /// This methods checks if a ReportView with the same ReportData (content compare not reference compare!) was already added.
        /// From IReportData the two Properties ReportLocation and Parameters are checked. Needed for adding drilldowns.
        /// </summary>
        /// <param name="newData">The ReportData as source</param>
        /// <returns>The report view that matches the IReportData. Null if no match was found</returns>
        public IReportView GetView(IRenderData newData)
        {
            foreach (ReportView view in viewToPageCollection.Keys)
            {

                if (newData.Equals(view.ReportData))
                {
                    return view;
                }
            }
            return null;
        }

        /// <summary>
        /// Removes this view form this ReportViewer.       
        /// All Reports can be removed. Unlike in the UserInterface it is not possible to close the last remaining tab        
        /// </summary>
        /// <param name="view">The <see cref="Inet.Viewer.IReportView"/> that was added before</param>
        public void RemoveReportView(IReportView view)
        {
            if (ReportViews.Remove(view))
            {
                // remove view and page from Dictionary
                TabPage page = viewToPageCollection[view];
                viewToPageCollection.Remove(view);
                pageToViewCollection.Remove(page);
                ((ReportView)view).OnRemoved();

                this.tabControl1.TabPages.Remove(page);

                //set new CurrentView
                if (ReportViews.Count > 0)
                {
                    CurrentReportView = ReportViews[ReportViews.Count - 1];
                }
                else
                {
                    CurrentReportView = null;
                }
            }
        }

        /// <summary>
        /// If one of the tabs was changed set the ReportView for the selected tab
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void tabControl1_SelectedIndexChanged(object sender, EventArgs e)
        {
            TabPage selectedPage = this.tabControl1.SelectedTab;
            if (selectedPage != null)
            {
                bool containsPage = pageToViewCollection.ContainsKey(selectedPage);
                // code for .net >= 3.5 pageToViewCollection.Keys.Contains(selectedPage)                
                if (containsPage)
                {
                    IReportView selectedView = pageToViewCollection[selectedPage];
                    CurrentReportView = selectedView;

                    // Manually set the focus to the PageView. Without this the focus would be on the tab-headers
                    if (selectedPage.Controls.ContainsKey(ReportViewPanelName))
                    {
                        ReportView viewPanel = selectedPage.Controls[ReportViewPanelName] as ReportView;
                        if (viewPanel != null)
                        {
                            viewPanel.FocusContent();
                        }
                    }
                }
                else
                {
                    CurrentReportView = null;
                }
            }
            else
            {
                CurrentReportView = null;
            }
        }

        /// <summary>        
        /// Checks if the mouse click was on the tab item close icon.
        /// If yes than close the tab on which was clicked on.
        /// </summary>
        /// <remarks>Only <see cref="Inet.Viewer.IReportView.IsCloseable"/> closable RepoertViews can be closed</remarks>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void tabControl1_MouseClick(object sender, MouseEventArgs e)
        {
            //Looping through all Tabs 
            for (int i = 0; i < this.tabControl1.TabPages.Count; i++)
            {
                TabPage page = this.tabControl1.TabPages[i];
                if (pageToViewCollection.ContainsKey(page))
                {
                    IReportView view = pageToViewCollection[page];
                    // check if this view is closeable
                    if (view.IsCloseable)
                    {
                        Rectangle r = tabControl1.GetTabRect(i);
                        //Getting the position of the "x" mark of the selected Tab
                        Rectangle closeButton = new Rectangle(r.Right - TabCloseSelectedRight + 2, r.Top + TabCloseSelectedTop, TabCloseImageSize, TabCloseImageSize - 1);
                        if (closeButton.Contains(e.Location))
                        {
                            this.RemoveReportView(view);
                            break;
                        }
                    }
                }
            }
        }

        /// <summary>
        /// Draws one tabItem with a close icon if the <seealso cref="Inet.Viewer.IReportView"/> Property <seealso cref="Inet.Viewer.IReportView.IsCloseable"/> is
        /// set to true.
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void tabControl1_DrawItem(object sender, DrawItemEventArgs e)
        {
            TabControl tabControl = (TabControl)sender;
            TabPage page = tabControl.TabPages[e.Index];
            if (pageToViewCollection.ContainsKey(page))
            {
                IReportView view = pageToViewCollection[page];
                if (view.IsCloseable)
                {
                    // This code will render a "x" mark at the end of the Tab caption.            
                    if (e.State == DrawItemState.Selected)
                    {
                        // selected
                        if (closeButton != null)
                        {
                            e.Graphics.DrawImage(closeButton, e.Bounds.Right - TabCloseSelectedRight, e.Bounds.Top + TabCloseSelectedTop, closeButton.Width, closeButton.Height);
                        }
                    }
                    else
                    {
                        // all other states                    
                        if (closeButtonDisabled != null)
                        {
                            e.Graphics.DrawImage(closeButtonDisabled, e.Bounds.Right - 18, e.Bounds.Top + 3, closeButtonDisabled.Width, closeButtonDisabled.Height);
                        }
                    }
                }
                e.Graphics.DrawString(this.tabControl1.TabPages[e.Index].Text, e.Font, Brushes.Black, e.Bounds.Left + 4, e.Bounds.Top + 4);
                e.DrawFocusRectangle();
            }
        }


        #region implement interface IReportViewer
        /// <summary>
        /// <inheritdoc/>
        /// </summary>
        public int ReportViewCount
        {
            get
            {
                return ReportViews.Count;
            }
        }


        /// <summary>
        /// <inheritdoc/>
        /// </summary>
        public IReportView GetReportView(int i)
        {
            return ReportViews[i];
        }

        /// <summary>
        /// <inheritdoc/>
        /// </summary>
        public IToolBar ToolBar
        {
            get
            {
                return viewerToolbar1;
            }
        }

        /// <summary>
        /// <inheritdoc/>
        /// </summary>
        public IReportView AddNewReportView(IRenderData data)
        {
            IReportView viewAlreadyExists = GetView(data);
            if (viewAlreadyExists != null)
            {
                this.CurrentReportView = viewAlreadyExists;
                return CurrentReportView;
            }
            else
            {
                IReportView view = new ReportView();
                view.ReportData = data;
                AddReportView(view);
                return view;
            }

        }

        /// <summary>
        /// <inheritdoc/>
        /// </summary>
        public void CloseReportView(int index)
        {
            IReportView view = ReportViews[index];
            RemoveReportView(view);
        }

        /// <summary>
        /// <inheritdoc/>
        /// </summary>
        public void CloseReportView(IReportView view)
        {
            RemoveReportView(view);
        }

        /// <summary>
        /// <inheritdoc/>
        /// </summary>
        public void CloseAllReportViews()
        {
            for (int i = ReportViews.Count - 1; i >= 0; i--)
            {
                CloseReportView(i);
            }
        }

        /// <summary>
        /// <inheritdoc/>
        /// </summary>
        public event EventHandler ViewChanged;
        #endregion
        
        /// <inheritdoc />
        public bool SearchVisible
        {
            get
            {
                return !splitContainer1.Panel2Collapsed;
            }
            set
            {
                if (reportViews.Count == 0)
                {
                    value = false;
                }
                splitContainer1.Panel2Collapsed = !value;
                if (value)
                {
                    searchPanel.FocusInputField();
                }
                else
                {
                    foreach (ReportView reportView in reportViews)
                    {
                        reportView.ClearSelection();
                    }
                }
            }
        }

        /// <summary>
        /// Called when the user clicks in the close button of the search panel. Synchronizes the state with the checked flag of the toolbar. 
        /// </summary>
        /// <param name="sender">the sender</param>
        /// <param name="e">the event arguments</param>
        private void searchPanel_CloseClick(object sender, EventArgs e)
        {
            SearchVisible = false;
            ToolBar.SearchChecked = false;
        }

        /// <inheritdoc/>
        public System.Drawing.Printing.PrinterSettings PrinterSettings
        {
            get
            {
                return printerSettings;
            }
        }

        /// <summary>
        /// Gets the export dialog.
        /// </summary>
        internal ExportDialog ExportDialog
        {
            get
            {
                return exportDialog;
            }
        }
    }

}








